unit fCommodore;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, FileCtrl, RzFilSys, StdCtrls, Menus, Gauges;

type
  TfCommodore64 = class(TForm)
    Bevel1: TBevel;
    Bevel2: TBevel;
    MainMenu1: TMainMenu;
    mFile: TMenuItem;
    barD64: TGauge;
    barZip: TGauge;
    mAbout: TMenuItem;
    mExit: TMenuItem;
    pnlD64Back: TPanel;
    lstDirD64: TRzDirectoryListBox;
    lstFilesD64: TRzFileListBox;
    btnD64: TButton;
    pnlD64: TPanel;
    cbxDriveD64: TRzDriveComboBox;
    pnlD64Header: TPanel;
    Label1: TLabel;
    pnlZipBack: TPanel;
    lstDirZip: TRzDirectoryListBox;
    lstFilesZip: TRzFileListBox;
    pnlZip: TPanel;
    cbxDriveZip: TRzDriveComboBox;
    Panel2: TPanel;
    Label3: TLabel;
    btnZip: TButton;
    imgCommodore: TImage;
    procedure btnD64Click(Sender: TObject);
    procedure btnZipClick(Sender: TObject);
    procedure mExitClick(Sender: TObject);
    procedure mAboutClick(Sender: TObject);
  private
    vTrackInfo: Array [0..45000] of Byte; {Array to hold each zip file's bytes}
    vDiskImage: Array [0..180000] of Byte;{Array to hold entire disk image}
    vPos: Integer;                        {Current position within a file}
    vTrackData: Array [0..8192] of Byte;  {Array to hold the actual disk info}
    vTotalBytes: Integer;                 {Keeps track of total bytes per zip file}
    procedure CheckData(vTrack: Integer; vMaxSec: Integer);
    function CompressData(vTrack: Integer; vSec: Byte; vMaxSec: Integer; vType: Byte): Integer;
  public
    { Public declarations }
  end;

var
  fCommodore64: TfCommodore64;

implementation

{$R *.DFM}

procedure TfCommodore64.btnD64Click(Sender: TObject);
var
  OutFileName: String;     {Holds the output file name}
  vTrack: Byte;            {Holds the current track number}
  vSector: Byte;           {Holds the current sector number}
  vMaxSectors: Byte;       {Keeps track of the max # sectors per track number}
  FInFile, FOutFile: TFileStream; {I/O file objects}
  vTrk: Byte;           {Holds the compression type and the track number}
  vSec: Byte;           {Holds the current sector we are reading}
  vLen: Byte;           {How many bytes before we reach the next sector}
  vRep: Byte;           {Variable to check for repetition}
  vRepNum: Byte;        {How many times a byte is repeated}
  vChar: Byte;          {The actual byte read - what we write to the disk}
  vCount: Integer;      {Holds our position within vTrackData array}
  vCurPos: Integer;     {Count to cycle through the bytes we are reading}
  vRepCounter: Integer; {Counter for looping through repeating bytes}
  vDiskPos: Integer;    {Where we are in the disk image}
begin
  try
    Screen.Cursor := crHourGlass;

    if lstFilesD64.FileName = '' then
      begin
        MessageBox(Handle, 'You must choose a file before converting.',
          'File Error', MB_ICONERROR);
        exit;
      end; {if}

    with lstFilesD64 do
      begin
        if not FileExists(FileName) then
          begin
            MessageBox(Handle, PChar('The file ' + FileName + ' does not exist.'),
              'File Error', MB_ICONERROR);
            exit;
          end {if}
        else
          begin
            {Create the output filename using the listbox selection}
            OutFileName := Copy(FileName, Length(Directory) + 4,
              Length(FileName) - (Length(Directory)) + 4);
          end; {else}
      end; {with}

    {Setup the progress bar}
    with barD64 do
      begin
        Progress := 0;
        Visible := True;
      end; {with}

    {Clear misc counters}
    vPos := 0;
    vDiskPos := 0;

    {Zero out the vDiskImage array}
    FillChar(vDiskImage, SizeOf(vDiskImage), 0);

    {Repeat code for each track - A standard Commodore disk has 35 tracks}
    for vTrack := 1 to 35 do
      begin
        {Zero out the vTrackData array}
        FillChar(vTrackData, SizeOf(vTrackData), 0);

        {Set the max number of sectors depending on which track we are on}
        case vTrack of
          01..17: vMaxSectors := 20;
          18..24: vMaxSectors := 18;
          25..30: vMaxSectors := 17;
          31..35: vMaxSectors := 16;
        end; {case}

        try
          {Attempt to open the input files as needed}
          case vTrack of
            01: begin
                  FInFile := TFileStream.Create('1!' + OutFileName, fmOpenRead);
                  FInFile.Seek(4, soFromBeginning);
                end;
            09: begin
                  barD64.Progress := barD64.Progress + 25; {Update the status bar}
                  FInFile.Free; {Destroy the old file object}
                  FInFile := TFileStream.Create('2!' + OutFileName, fmOpenRead);
                  FInFile.Seek(2, soFromBeginning);
                end;
            17: begin
                  barD64.Progress := barD64.Progress + 25; {Update the status bar}
                  FInFile.Free; {Destroy the old file object}
                  FInFile := TFileStream.Create('3!' + OutFileName, fmOpenRead);
                  FInFile.Seek(2, soFromBeginning);
                end;
            26: begin
                  barD64.Progress := barD64.Progress + 25; {Update the status bar}
                  FInFile.Free; {Destroy the old file object}
                  FInFile := TFileStream.Create('4!' + OutFileName, fmOpenRead);
                  FInFile.Seek(2, soFromBeginning);
                end;
          end; {case}
        except
          case vTrack of
            01..08: OutFileName := '1!' + OutFileName;
            09..16: OutFileName := '2!' + OutFileName;
            17..25: OutFileName := '3!' + OutFileName;
            26..35: OutFileName := '4!' + OutFileName;
          end; {case}

          MessageBox(Handle, PChar('Unable to open the file ' + OutFileName +
            '. The file may be in use by another application, corrupt, or ' +
            'does not exist.'), 'File Error', MB_ICONERROR);
          exit;
        end; {try}

        {Read the disk information for each sector on the current track}
        for vSector := 0 to vMaxSectors do
          begin
            try
              {Read the track/compression type}
              FInFile.Read(vTrk, 1);
              {Read which sector we are working with}
              FInFile.Read(vSec, 1);

              {Check to make sure all information is valid}
              if ((vTrk and 63) <> vTrack) or (vSec > vMaxSectors) then
                begin
                  MessageBox(Handle, 'The image file contains an invalid ' +
                    'track or sector number.', 'Track/Sector Error', MB_ICONERROR);
                  barD64.Visible := False;
                  FInFile.Free;
                  exit;
                end; {if}

              if ((vTrk and 128) = 128) then {Means the track has some repeating,}
                begin                        {concurrent bytes}
                  FInFile.Read(vLen, 1);     {Number of bytes before the next sector}
                  FInFile.Read(vRep, 1);     {Value to warn us when repeating bytes are next}

                  vCurPos := 0;
                  vCount := 0;

                  while vCurPos <= vLen - 1 do {Continue reading bytes until the next sector}
                    begin
                      {Make sure we don't try to read beyond the end of file}
                      if FInFile.Position + (vLen - vCurPos - 1) > FInFile.Size then
                        begin
                          MessageBox(Handle, 'The zip files contain a different ' +
                          'number of  bytes than they should actually have.',
                          'Zips Corrupt', MB_ICONERROR);
                          barD64.Visible := False;
                          FInFile.Free;
                          exit;
                        end;

                      FInFile.Read(vChar, 1); {Read the first data byte in the series}
                      if (vChar <> vRep) then {Is this a repeating value (vChra = vRep)?}
                        begin
                          {Write the unique byte to vTrackData}
                          vTrackData[vSec shl 8 + vCount] := vChar;
                          inc(vCount);
                          inc(vCurPos);
                        end {if}
                      else   {vChra = vRep so the next set of bytes are repeating}
                        begin
                          {Find how many repeating bytes there are}
                          FInFile.Read(vRepNum, 1);

                          FInFile.Read(vChar, 1); {Byte (value) to be repeated}
                          inc(vCurPos, 3);

                          for vRepCounter := 0 to vRepNum - 1 do
                            begin
                              {Write the repeating value to vTrackData vRepNum times}
                              vTrackData[vSec shl 8 + vCount] := vChar;
                              inc(vCount);
                            end;
                        end; {else}
                    end; {while}
                end {if}
              else if ((vTrk and 64) = 64) then  {The whole sector has the same value}
                begin
                  {Make sure we don't try to read beyond the end of file}
                  if FInFile.Position + (vLen - vCurPos - 1) > FInFile.Size then
                    begin
                      MessageBox(Handle, 'The zip files contain a different ' +
                        'number of  bytes than they should actually have.',
                        'Zips Corrupt', MB_ICONERROR);
                      barD64.Visible := False;
                      FInFile.Free;
                      exit;
                    end; {if}

                  FInFile.Read(vChar, 1); {Read the byte to be written}
                  FillChar(vTrackData[vSec shl 8], 256, vChar); {Write 256 bytes worth of}
                end {else if}
              else  {No concurrent bytes repeat - they are all unique}
                begin
                  {Read the next 256 bytes and place them in vTrackData}
                  FInFile.Read(vTrackData[vSec shl 8], 256);
                end; {else}
            except
              on EReadError do
                begin
                  MessageBox(Handle, 'An error occured while trying to read ' +
                    'from the image file.', 'Read Error', MB_ICONERROR);
                  exit;
                end; {on}
            end; {try}
          end; {for}

          {Write the track to the disk array}
            for vCount := 1 to (vMaxSectors * 256 + 256) do
              begin
                vDiskImage[vDiskPos] :=
                  vTrackData[vCount - 1]; {vTrackData is zero based}
                inc(vDiskPos);
              end; {for}
      end; {for}

    {Update the status bar to show we are finished}
    barD64.Progress := barD64.Progress + 25;

    try
      {Create the output file *.d64}
      FOutFile := TFileStream.Create(lstFilesD64.Directory + '\' + OutFileName +
        '.d64', fmCreate);

      {Write the current data (track) to the output file}
      FOutFile.Write(vDiskImage, 174848); {174848 = actual disk size in bytes}
    except
      on EFCreateError do
        begin
          MessageBox(Handle, 'An error occured while trying to create ' +
            'the disk image file.', 'Create Error', MB_ICONERROR);
          FInFile.Free;
          exit;
        end; {on}
      on EWriteError do
        begin
          MessageBox(Handle, 'An error occured while trying to write to the ' +
            'disk image file.', 'Write Error', MB_ICONERROR);
          FInFile.Free;
          FOutFile.Free;
          exit;
        end;
    end; {try}

    FInFile.Free;  {Destroy the last file object}
    FOutFile.Free; {Destroy the output file object}

    {Change filename so it isn't all capitalized}
    OutFileName := LowerCase(OutFileName);
    OutFileName[1] := UpCase(OutFileName[1]);
    RenameFile(lstFilesD64.Directory + '\' + OutFileName + '.d64',
      lstFilesD64.Directory + '\' + OutFileName + '.d64');

    {Let the user know it is finished}
    MessageBox(Handle, 'Image file created successfully.', 'Success', MB_OK +
      MB_ICONINFORMATION);
  finally
    {Remove the status bar}
    barD64.Visible := False;
    Screen.Cursor := crDefault;
    lstFilesZip.Update;
    lstFilesD64.Update;
  end;
end;

procedure TfCommodore64.btnZipClick(Sender: TObject);
var
  vTrack, vMaxSec, vHeader: Integer; {Track, sector, and zip header}
  OutFileName: String;               {Output filename}
  FInFile, FOutFile: TFileStream;    {I/O file objects}
begin
  try
    Screen.Cursor := crHourGlass;

    if lstFilesZip.FileName = '' then
      begin
        MessageBox(Handle, 'You must choose a file before converting.',
          'File Error', MB_ICONERROR);
        exit;
      end; {if}

    try
      {Open the disk image}
      FInFile := TFileStream.Create(lstFilesZip.FileName, fmOpenRead);
    except
      MessageBox(Handle, PChar('Unable to open the file ' + lstFilesZip.FileName +
        '. The file may be in use by another application, corrupt, or ' +
        'does not exist.'), 'File Error', MB_ICONERROR);
      exit;
    end; {try}

    {Check for a valid image size}
    if FInFile.Size <> 174848 then
      begin
        MessageBox(Handle, PChar('The file ' + lstFilesZip.FileName + ' does ' +
          'not contain the correct number of bytes for a valid disk image.'),
          'Invalid Image', MB_ICONERROR);
        FInFile.Free;
        exit;
      end; {if}

    {Clear necessary variables}
    FillChar(vTrackInfo, SizeOf(vTrackInfo), 0);
    FillChar(vTrackData, SizeOf(vTrackData), 0);
    vTotalBytes := 0;
    vPos := 0;

    {Setup the progress bar}
    with barZip do
      begin
        Progress := 0;
        Visible := True;
      end; {with}

   with lstFilesZip do
      begin
        {Construct the output file name}
        OutFileName := Copy(FileName, Length(Directory) + 2,
          Length(FileName) - Length(Directory) - 5);
      end; {with}

    for vTrack := 1 to 35 do
      begin
        case vTrack of
          01..17: vMaxSec := 20;
          18..24: vMaxSec := 18;
          25..30: vMaxSec := 17;
          31..35: vMaxSec := 16;
        end; {case}

        try
          FInFile.Read(vTrackData, vMaxSec * 256 + 256); {Read one full track}
        except
          MessageBox(Handle, 'An error occurec while trying to read from ' +
            'the disk image file.', 'Read Error', MB_ICONERROR);
          FInFile.Free;
          exit;
        end; {try}

        {Prepare the data for compression}
        CheckData(vTrack, vMaxSec);

        try
          case vTrack of
            08: begin
                  {The first zip file ends with track 8}
                  FOutFile := TFileStream.Create('1!' + OutFileName, fmCreate);
                  vHeader := $343603FE;
                  with FOutFile do
                    begin
                      Write(vHeader, 4); {Write the four byte header}
                      Write(vTrackInfo, vTotalBytes);
                      Free;
                    end; {with}
                  vTotalBytes := 0; {Reset total for next tracks}
                  vPos := 0;
                  FillChar(vTrackInfo, SizeOf(vTrackInfo), 0);
                end;
            16: begin
                  {The second zip file ends with track 16}
                  barZip.Progress := 25;
                  FOutFile := TFileStream.Create('2!' + OutFileName, fmCreate);
                  vHeader := $0400;
                  with FOutFile do
                    begin
                      Write(vHeader, 2); {Write the first two bytes}
                      Write(vTrackInfo, vTotalBytes);
                      Free;
                    end;
                  vTotalBytes := 0;
                  vPos := 0;
                  FillChar(vTrackInfo, SizeOf(vTrackInfo), 0);
                end;
            25: begin
                  {The third zip file ends with track 25}
                  barZip.Progress := 50;
                  FOutFile := TFileStream.Create('3!' + OutFileName, fmCreate);
                  vHeader := $0400;
                  with FOutFile do
                    begin
                      Write(vHeader, 2); {Write the first two bytes}
                      Write(vTrackInfo, vTotalBytes);
                      Free;
                    end;
                  vTotalBytes := 0;
                  vPos := 0;
                  FillChar(vTrackInfo, SizeOf(vTrackInfo), 0);
                end;
            35: begin
                  {The fourth zip file ends with track 35}
                  barZip.Progress := 75;
                  FOutFile := TFileStream.Create('4!' + OutFileName, fmCreate);
                  vHeader := $0400;
                  with FOutFile do
                    begin
                      Write(vHeader, 2); {Write the first two bytes}
                      Write(vTrackInfo, vTotalBytes);
                      Free;
                    end;
                  vTotalBytes := 0;
                  FillChar(vTrackInfo, SizeOf(vTrackInfo), 0);
                end;
          end; {case}
        except
          on EFCreateError do
            begin
              MessageBox(Handle, 'An error occured while trying to create ' +
                'one of the output zip files.', 'File Error', MB_ICONERROR);
              FInFile.Free;
              exit;
            end; {on}
          on EWriteError do
            begin
              MessageBox(Handle, 'An error occured while trying to write ' +
                'to one of the output zip files.', 'Write Error', MB_ICONERROR);
              FOutFile.Free;
              FInFile.Free;
              exit;
            end; {on}
        end; {try}
      end; {for}
    barZip.Progress := 100;

    FInFile.Free;

    {Let the user know it is finished}
    MessageBox(Handle, 'Zip files created successfully.', 'Success', MB_OK);
  finally
    {Remove the status bar}
    barZip.Visible := False;
    Screen.Cursor := crDefault;
    lstFilesZip.Update;
    lstFilesD64.Update;
  end;
end;

procedure TfCommodore64.CheckData(vTrack: Integer; vMaxSec: Integer);
var
  vSec: Integer;      {The current sector}
  vArrayPos: Integer; {Position within a sector of bytes}
  vHalfway: Integer;  {Midway point}
  vRep: Integer;      {Counts how many concurrent bytes repeat}
  vType: Byte;        {Compression type}
  vSubTot: Integer;   {Keeps track of how many total bytes repeat}
begin
  vHalfway := (vMaxSec div 2) + 1;

  {This section checks the lower half of the sectors in the track}
  for vSec := 0 to (vMaxSec div 2) do
    begin
      vType := 0;  {Assume no compression}
      vArrayPos := 0;
      vRep := 0;
      vSubTot := 0;
      while vArrayPos < 255 do
        begin
          if vTrackData[vSec shl 8 + vArrayPos] =
            vTrackData[vSec shl 8 + vArrayPos + 1] then
            begin
              inc(vRep);

              if vRep = 5 then vType := 1  {6 matching bytes}
              else if vRep = 255 then vType := 2;

              if ((vArrayPos = 254) and (vRep in [3, 4])) then
                inc(vSubTot, vRep - 2);

              inc(vArrayPos);
            end {if}
          else
            begin
              inc(vArrayPos);
              if ((vType = 0) and (vRep in [3, 4])) then
                begin
                  {Run a subtotal}
                  inc(vSubTot, vRep - 2);
                end; {if}
              vRep := 0;
            end; {else}
        end; {while}

      {Do we have enough sets of repeating bytes to compress}
      if ((vSubTot >= 3) and (vType = 0)) then
        vType := 1;

      CompressData(vTrack, vSec, vMaxSec, vType);

      {This section checks the upper half of the sectors in the track}
      case vMaxSec of
        20:  begin
               {Last sector in track to be processed}
               if vSec = 10 then
                 exit;
             end;
        18:  begin
               {Last sector in track to be processed}
               if vSec = 9 then
                 exit;
             end;
        16:  begin
               {Last sector in track to be processed}
               if vSec = 8 then
                 exit;
             end;
      end; {case}

      vType := 0;  {Assume no compression}
      vArrayPos := 0;
      vRep := 0;
      vSubTot := 0;
      while vArrayPos < 255 do
        begin
          if vTrackData[(vSec + vHalfway) shl 8 + vArrayPos] =
            vTrackData[(vSec + vHalfway) shl 8 + vArrayPos + 1] then
            begin
              inc(vRep);

              if vRep = 5 then vType := 1  {6 matching bytes}
              else if vRep = 255 then vType := 2;

              if ((vArrayPos = 254) and (vRep in [3, 4])) then
                inc(vSubTot, vRep - 2);

              inc(vArrayPos);
            end {if}
          else
            begin
              inc(vArrayPos);
              if ((vType = 0) and (vRep in [3, 4])) then
                begin
                  {Run a subtotal}
                  inc(vSubTot, vRep - 2);
                end; {if}
              vRep := 0;
            end; {else}
        end; {while}

      if ((vSubTot >= 3) and (vType = 0)) then
        vType := 1;

      CompressData(vTrack, vSec + vHalfway, vMaxSec, vType);
    end; {for}
end;

function TfCommodore64.CompressData(vTrack: Integer; vSec: Byte; vMaxSec: Integer; vType: Byte): Integer;
var
  vCount: Integer;                    {General counter}
  vFreeByte: Array [0..255] of Byte;  {Keeps track of which bytes isn't in the sector}
  vRep, vRepValue: Byte;              {# of repeating bytes and which byte repeats}
  vPlaceHolder: Integer;              {Where vNumBytes is stored}
  vNumBytes: Integer;                 {Number of bytes in sector}
begin
  case vType of
    0: begin {No bytes match}
         vTrackInfo[vPos] := vTrack;
         inc(vPos);
         vTrackInfo[vPos] := vSec;
         inc(vPos);

         for vCount := 0 to 255 do
            begin
              vTrackInfo[vPos] := vTrackData[vSec shl 8 + vCount];
              inc(vPos);
            end; {for}
         inc(vTotalBytes, 258);
       end;
    1: begin {Some bytes match}
         vNumBytes := 0;

         vTrackInfo[vPos] := vTrack or 128;
         inc(vPos);
         vTrackInfo[vPos] := vSec;
         inc(vPos, 2); {Save space for number of bytes until next sector}
         vPlaceHolder := vPos - 1; {Actual place for number of bytes}
         inc(vNumBytes, 2);
         inc(vTotalBytes, 2);

         {Scan for a free byte}
         FillChar(vFreeByte, SizeOf(vFreeByte), 0);
         for vCount := 0 to 255 do
           vFreeByte[vTrackData[vSec shl 8 + vCount]] := 1;

         {Assign the free byte}
         for vCount := 0 to 255 do
           begin
             if vFreeByte[vCount] = 0 then
               begin
                 {This is our repeating value}
                 vTrackInfo[vPos] := vCount;
                 vRepValue := vCount;
                 inc(vPos);
                 inc(vNumBytes);
                 inc(vTotalBytes);
                 break;
               end; {if}
           end;

         {Start compressing}
         vCount := 0;
         vRep := 0;
         while vCount < 255 do
           begin
             vRep := 0;
             if vTrackData[vSec shl 8 + vCount] =
               vTrackData[vSec shl 8 + vCount + 1] then
               begin
                 inc(vRep, 2); {Two matching bytes}
                 inc(vCount);

                 while vCount < 255 do
                   begin
                     if vTrackData[vSec shl 8 + vCount] =
                       vTrackData[vSec shl 8 + vCount + 1] then
                         begin
                           inc(vRep);
                           inc(vCount);
                         end {if}
                     else
                       begin
                         break;
                       end; {else}
                   end; {While}

                 if vRep >= 4 then
                   begin
                     {This is the repeat notifier}
                     vTrackInfo[vPos] := vRepValue;
                     inc(vPos);
                     {This is the number of repeating bytes}
                     vTrackInfo[vPos] := vRep;
                     inc(vPos);
                     {This is the actual repeating byte}
                     vTrackInfo[vPos] := vTrackData[vSec shl 8 + vCount];
                     inc(vPos);
                     inc(vNumBytes, 3);
                     inc(vTotalBytes, 3);
                     vRep := 0;

                     if ((vCount = 254) and (vTrackData[vSec shl 8 + vCount] <>
                       vTrackData[vSec shl 8 + vCount + 1])) then
                       begin
                         vTrackInfo[vPos] := vTrackData[vSec shl 8 + 255];
                         inc(vPos);
                         inc(vNumBytes);
                         inc(vTotalBytes);
                       end;

                     inc(vCount);
                   end {if}
                 else
                   begin
                     dec(vCount, vRep - 1);
                     for vRep := 1 to vRep do
                       begin
                         vTrackInfo[vPos] := vTrackData[vSec shl 8 + vCount];
                         inc(vPos);
                         inc(vNumBytes);
                         inc(vTotalBytes);
                         if ((vCount = 254) and (vTrackData[vSec shl 8 + vCount] <>
                           vTrackData[vSec shl 8 + vCount + 1])) then
                           begin
                             vTrackInfo[vPos] := vTrackData[vSec shl 8 + 255];
                             inc(vPos);
                             inc(vNumBytes);
                             inc(vTotalBytes);
                           end; {if}

                         inc(vCount);
                       end; {for}
                   end; {else}
               end {if}
             else
               begin
                 vTrackInfo[vPos] := vTrackData[vSec shl 8 + vCount];
                 inc(vPos);
                 inc(vCount);
                 inc(vNumBytes);
                 inc(vTotalBytes);
                 if vCount = 255 then
                   begin
                     vTrackInfo[vPos] := vTrackData[vSec shl 8 + vCount];
                     inc(vPos);
                     inc(vNumBytes);
                     inc(vTotalBytes);
                   end; {if}
                 vRep := 0;
               end; {else}
           end; {while}

         vTrackInfo[vPlaceHolder] := vNumBytes - 3;
         inc(vTotalBytes);
       end;
    2: begin {All bytes match}
         vTrackInfo[vPos] := vTrack or 64;
         inc(vPos);
         vTrackInfo[vPos] := vSec;
         inc(vPos);
         vTrackInfo[vPos] := vTrackData[vSec shl 8]; {Pick the first byte}
         inc(vPos);
         inc(vTotalBytes, 3);
       end;
  end; {Case}
end;

procedure TfCommodore64.mExitClick(Sender: TObject);
begin
  Close;
end;

procedure TfCommodore64.mAboutClick(Sender: TObject);
begin
  MessageBox(Handle, 'Commodore Conversion Utility' + #13#13 + 'Created by ' +
    'Cyberonix in 1999' + #13#13 + '    cyberonix@hotmail.com',
    'Commodore Conversion Utility v1.0', MB_OK + MB_ICONINFORMATION);
end;


end.
